/**
 * \file: libsugc_set.c
 *
 * \brief : Functions to perform actual configuration
 *
 * \author: Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#define _GNU_SOURCE
#include <stdbool.h>
#include <string.h>
#include <sys/prctl.h>
#include <linux/securebits.h>
#include <sys/capability.h>
#include <errno.h>
#include <unistd.h>
#include <grp.h>
#include "libsugc_set.h"
#include "libsugc_log.h"

/**
 * \ingroup libsugc_set_api
 * \brief UID of the root user
 */
#define ROOT_UID ((uid_t)0)

/**
 * \ingroup libsugc_set_api
 * \brief get capabilities of current process
 *
 * \param[out] curr_caps    Pointer to return capabilities
 *
 * \return \ref libsugc_error_t
 */
static libsugc_error_t get_current_capabilities(cap_t *curr_caps)
{
    *curr_caps = cap_get_proc();
    if (curr_caps == NULL) {
        LIBSUGC_ERROR("Failed to get capabilities of current process");
        return LIBSUGC_FAILED;
    }

    return LIBSUGC_OK;
}

/**
 * \ingroup libsugc_set_api
 * \brief set capabilities of current process
 *
 * \param[in] set_caps      Capabilities to set
 *
 * \return \ref libsugc_error_t
 */
static libsugc_error_t set_capabilities(const cap_t set_caps)
{
    int res;

    LIBSUGC_DEBUG("Setting capabilities");

    errno = 0;
    res = cap_set_proc(set_caps);
    if (res != 0) {
        LIBSUGC_ERROR("Failed to set caps of process (%d, %s)",
              res, strerror(errno));
        return LIBSUGC_FAILED;
    }

    return LIBSUGC_OK;
}

/**
 * \ingroup libsugc_set_api
 * \brief free capability data structure
 *
 * \param[in] caps      capability data structure to free
 *
 * \return \ref libsugc_error_t
 */
static libsugc_error_t free_capabilities(cap_t caps)
{
    libsugc_error_t err = LIBSUGC_OK;
    int res;

    res = cap_free(caps);
    if (res < 0) {
        LIBSUGC_ERROR("cap_free failed");
        err = LIBSUGC_FAILED;
    }

    return err;
}

/**
 * \ingroup libsugc_set_api
 * \brief Try to add setuid capability
 *
 * In order to switch user and groups to arbitrary values CAP_SETUID is required
 * In case this capability is currently missing in the effective capabilities
 * these syscalls would fail.
 * So try to add CAP_SETUID in effective capabilities - this will only work if
 * CAP_SETUID is part of the permitted capabilities.
 *
 * \return \ref libsugc_error_t
 */
static libsugc_error_t add_setuid_capability(void)
{
    libsugc_error_t err;
    libsugc_error_t tmperr;
    cap_t caps = NULL;
    cap_value_t cap_list = CAP_SETUID;
    int res;

    err = get_current_capabilities(&caps);

    if (err == LIBSUGC_OK) {
        res = cap_set_flag(caps, CAP_EFFECTIVE, 1, &cap_list, CAP_SET);
        if (res != 0) {
            LIBSUGC_ERROR("Failed to set CAP_SETUID flag");
            err = LIBSUGC_FAILED;
        }
    }

    if (err == LIBSUGC_OK)
        err = set_capabilities (caps);

    tmperr = free_capabilities(caps);
    if (err == LIBSUGC_OK)
        err = tmperr;

    return err;
}

/**
 * \ingroup libsugc_set_api
 * \brief Set effective, inheritable and permitted capabilities to 0
 *
 * \return \ref libsugc_error_t
 */
static void drop_all_capabilities(void)
{
    cap_t no_caps;
    int res;

    /* clear all capabilities */
    no_caps = cap_init();
    if (no_caps == NULL)
        LIBSUGC_ERROR("cap_init failed while trying to drop all capabilities");
    /* even if cap_init failed we will try to clear caps*/
    errno = 0;
    res = cap_set_proc(no_caps);
    if (res != 0)
        LIBSUGC_ERROR("cap_set_proc failed while trying to drop all capabilities (%d, %s)",
              res, strerror(errno));

    (void)free_capabilities(no_caps);
}

/**
 * \ingroup libsugc_set_api
 * \brief Check if we need to set SECBIT_KEEP_CAPS to avoid that all capabilities are dropped when switching to non-root
 *
 * \param[in] set_caps          The capabilities which will be set later
 * \param[in] no_caps           Capability struct having no capabilities set - needed for comparison
 * \param[in] stay_root         true if the final user will still be root - false when setting to non-root user
 * \param[out] need_set_caps    Pointer to return true if setting the capabilities is still required.
 *                              In case switching to non-root user and \p set_caps == \p no_caps all capabilities will
 *                              be dropped automatically when switching the user to non-root (SECBIT_KEEP_CAPS not set)
 *
 * \return \ref libsugc_error_t
 */
static libsugc_error_t prepare_cap_config(const cap_t set_caps,
                                          const cap_t no_caps,
                                          bool stay_root,
                                          bool *need_set_caps)
{
    libsugc_error_t err;
    libsugc_error_t tmperr;
    int res;

    cap_t curr_caps = NULL;
    bool has_caps = false;

    *need_set_caps = true;

    LIBSUGC_DEBUG("Prepare capability configuration");

    err = get_current_capabilities(&curr_caps);

    if (err == LIBSUGC_OK) {
        res = cap_compare(no_caps, set_caps);
        if (res < 0) {
            LIBSUGC_ERROR("Compare of no_caps to set_caps failed");
            err = LIBSUGC_FAILED;
        } else if (res != 0) {
            has_caps = true;
        }

        res = cap_compare(curr_caps, set_caps);
        if (res < 0) {
            LIBSUGC_ERROR("Compare of curr_caps to set_caps failed");
            err = LIBSUGC_FAILED;
        } else if (res == 0) {
            *need_set_caps = false;
        }
    }

    /*
     * we need to set the SECBIT_KEEP_CAPS in order not to discard
     * all capabilities when setting non root user
     */
    if (err == LIBSUGC_OK) {
        if ((!stay_root) &&
            (has_caps))
        {
            LIBSUGC_DEBUG("Set SECBIT_KEEP_CAPS");
            errno = 0;
            res = prctl(PR_SET_SECUREBITS, SECBIT_KEEP_CAPS);
            if (res < 0) {
                LIBSUGC_ERROR("set SECBIT_KEEP_CAPS failed (%d, %s)",
                              res, strerror(errno));
                err = LIBSUGC_FAILED;
            }
        } else {
            LIBSUGC_DEBUG("clear SECBIT_KEEP_CAPS");
            errno = 0;
            res = prctl(PR_SET_SECUREBITS, 0);
            if (res < 0) {
                LIBSUGC_ERROR("clear SECBIT_KEEP_CAPS failed (%d, %s)",
                              res, strerror(errno));
                err = LIBSUGC_FAILED;
            }

            if (!stay_root) {
                /* as !stay_root - has_caps needs to be false
                 * Therefore we switch to no capabilities and non-root
                 * all capabilities will implicitly be dropped when switching the user
                 * No explicit setting needed */
                *need_set_caps = false;
            }
        }
    }

    tmperr = free_capabilities(curr_caps);
    if (err == LIBSUGC_OK)
        err = tmperr;

    return err;
}

static libsugc_error_t set_user_and_groups_config(const libsugc_config_t *config)
{
    libsugc_error_t err = LIBSUGC_OK;
    int res;
    uid_t uid = config->uid;
    gid_t gid = config->gid;

    LIBSUGC_DEBUG("Setting user/group/supplementary groups");

    /* set supplementary groups */
    if (config->num_supp_grps != 0) {
        errno = 0;
        res = setgroups(config->num_supp_grps, config->supp_grps);
        if (res != 0) {
            LIBSUGC_ERROR("Failed to set supplementary groups (%d, %s)",
                          res, strerror(errno));
            err = LIBSUGC_FAILED;
        }
    }

    if (err == LIBSUGC_OK) {
        /* set real,saved,effective group */
        errno = 0;
        res = setresgid(gid, gid, gid);
        if (res != 0) {
            LIBSUGC_ERROR("Failed to set real, effective, saved groups (%d, %s)",
                          res, strerror(errno));
            err = LIBSUGC_FAILED;
        }
    }

    if (err == LIBSUGC_OK) {
        /* set real,saved,effective user */
        errno = 0;
        res = setresuid(uid, uid, uid);
        if (res != 0) {
            LIBSUGC_ERROR("Failed to set real, effective, saved user (%d, %s)",
                          res, strerror(errno));
            err = LIBSUGC_FAILED;
        }
    }

    return err;
}

void libsugc_set_fallback_config(void)
{
    uid_t uid = FALLBACK_UID;
    gid_t gid = FALLBACK_GID;
    int res;

    LIBSUGC_DEBUG("Set fallback config");

    /* just in case setuid capabilitiy is currently not in effective capabilities, but in permitted */
    (void)add_setuid_capability();

    /* set supplementary groups to primary group */
    errno = 0;
    res = setgroups(1, &gid);
    if (res != 0)
        LIBSUGC_ERROR("setgroups failed while trying to set fallback config  (%d, %s)",
                      res, strerror(errno));

    /* set primary group */
    errno = 0;
    res = setresgid(gid, gid, gid);
    if (res != 0)
        LIBSUGC_ERROR("setresgid failed while trying to set fallback config  (%d, %s)",
                      res, strerror(errno));

    /* set user */
    errno = 0;
    res = setresuid(uid, uid, uid);
    if (res != 0)
        LIBSUGC_ERROR("setresuid failed while trying to set fallback config  (%d, %s)",
                      res, strerror(errno));

    drop_all_capabilities();
}

libsugc_error_t libsugc_apply_config(const libsugc_config_t *config)
{
    libsugc_error_t err = LIBSUGC_OK;
    libsugc_error_t tmperr;
    cap_t no_caps = NULL;
    bool need_to_set_caps = false;
    const cap_t *cap_to_set;

    LIBSUGC_DEBUG("Set config");

    if (config == 0)
        return LIBSUGC_INVALID_PARAMETER;

    no_caps = cap_init();
    if (no_caps == NULL) {
        LIBSUGC_ERROR("Failed to initialize no_caps");
        err = LIBSUGC_FAILED;
    }

    /* cap struct used for cap_set_proc
     * used to point to no_caps if config->caps == NULL */
    cap_to_set = &no_caps;

    /* use no_caps only if no cap data in config */
    if (config->caps != NULL)
        cap_to_set = &(config->caps);

    if (err == LIBSUGC_OK)
        err = prepare_cap_config(*cap_to_set, no_caps,
                                 (config->uid == ROOT_UID),
                                 &need_to_set_caps);

    if (err == LIBSUGC_OK)
        err = set_user_and_groups_config(config);

    if (err == LIBSUGC_OK) {
        if (need_to_set_caps)
            err = set_capabilities(*cap_to_set);
        else
            LIBSUGC_DEBUG("No need to explicitly set capabilities");
    }

    tmperr = free_capabilities(no_caps);
    if (err == LIBSUGC_OK)
        err = tmperr;

    if (err == LIBSUGC_OK) {
        LIBSUGC_DEBUG("Set config succeeded");
    } else {
        LIBSUGC_INFO("Set config failed");
    }

    return err;
}
